# Architecture

## System Overview

```
┌─────────────────────────────────────────────────────────────┐
│                        Client (Browser)                      │
│  ┌──────────────┐  ┌──────────────┐  ┌──────────────┐     │
│  │   HTML/CSS   │  │  Turbo (JS)  │  │ ActionCable  │     │
│  │  (Tailwind)  │  │   (Hotwire)  │  │  (WebSocket) │     │
│  └──────────────┘  └──────────────┘  └──────────────┘     │
└─────────────────────────────────────────────────────────────┘
                              │
                   HTTP & WebSocket
                              │
┌─────────────────────────────────────────────────────────────┐
│                    Rails Application Server                  │
│  ┌──────────────────────────────────────────────────────┐  │
│  │                    Controllers                        │  │
│  │  • RoomsController  • MessagesController             │  │
│  │  • UsersController  • SessionsController             │  │
│  └──────────────────────────────────────────────────────┘  │
│  ┌──────────────────────────────────────────────────────┐  │
│  │                   Action Cable                        │  │
│  │  • RoomChannel  • PresenceChannel                    │  │
│  └──────────────────────────────────────────────────────┘  │
│  ┌──────────────────────────────────────────────────────┐  │
│  │                      Models                           │  │
│  │  • User  • Room  • Message                           │  │
│  └──────────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────────┘
                              │
              ┌───────────────┴───────────────┐
              │                               │
┌─────────────────────────┐    ┌─────────────────────────┐
│    PostgreSQL Database   │    │      Redis Cache        │
│  • users                 │    │  • Action Cable adapter │
│  • rooms                 │    │  • WebSocket pub/sub    │
│  • messages              │    │  • User presence        │
└─────────────────────────┘    └─────────────────────────┘
```

## Component Architecture

### 1. Presentation Layer (Views)

#### Room List Page
- Display all available rooms
- Show active user count per room
- Create new room form
- Navigation to sign out

#### Chat Room Page
- Message history (scrollable)
- Message input form
- Send button
- User list sidebar (who's online)
- Leave room button
- Auto-scroll to latest message

#### Authentication Pages
- Sign up form
- Sign in form
- Simple, minimal design

### 2. Controller Layer

#### RoomsController
```ruby
# Actions:
# - index: list all rooms
# - show: display chat room with messages
# - create: create new room
# - destroy: delete room (optional)
```

#### MessagesController
```ruby
# Actions:
# - create: post new message (broadcasts via cable)
```

#### UsersController
```ruby
# Actions:
# - new: sign up form
# - create: register new user
```

#### SessionsController
```ruby
# Actions:
# - new: sign in form
# - create: authenticate user
# - destroy: sign out
```

### 3. Model Layer

#### User Model
```ruby
# Attributes: username, password_digest
# Methods: authenticate (via bcrypt)
# Associations: has_many :messages
```

#### Room Model
```ruby
# Attributes: name, description
# Associations: has_many :messages
# Methods: broadcast_message, active_users_count
```

#### Message Model
```ruby
# Attributes: content, user_id, room_id, created_at
# Associations: belongs_to :user, belongs_to :room
# Validations: presence of content
```

### 4. Action Cable Layer

#### RoomChannel
```ruby
# Purpose: Handle real-time messaging
# Subscription: User subscribes to a room
# Actions:
#   - speak: broadcast message to all in room
#   - receive: handle incoming messages
```

#### PresenceChannel
```ruby
# Purpose: Track online users
# Actions:
#   - appear: user joined room
#   - disappear: user left room
#   - broadcast_presence: notify all users
```

## Data Flow

### Message Sending Flow

```
1. User types message and clicks Send
                    │
2. JavaScript captures submit event
                    │
3. Turbo submits form via fetch (Ajax)
                    │
4. MessagesController#create receives request
                    │
5. Create Message record in database
                    │
6. Broadcast via RoomChannel
                    │
7. Action Cable pushes to all subscribers
                    │
8. All clients receive via WebSocket
                    │
9. Stimulus controller appends message to DOM
                    │
10. Auto-scroll to bottom
```

### User Presence Flow

```
1. User joins room (opens chat page)
                    │
2. JavaScript subscribes to RoomChannel
                    │
3. Channel broadcasts "user_joined" event
                    │
4. All subscribers receive notification
                    │
5. Update online user list in UI
                    │
6. User leaves (closes page/navigates away)
                    │
7. WebSocket disconnect triggers "user_left"
                    │
8. Broadcast to remaining users
                    │
9. Update user list UI
```

## WebSocket Architecture

### Connection Lifecycle

```
Client                          Server (Action Cable)
  │                                      │
  │─────── Establish WebSocket ──────────→
  │                                      │
  │←────── Connection confirmed ─────────│
  │                                      │
  │─────── Subscribe to channel ─────────→
  │                                      │
  │←────── Subscription confirmed ───────│
  │                                      │
  │─────── Send message ──────────────────→
  │                                      │
  │←────── Broadcast to all ─────────────│
  │                                      │
  │       (message appears for all)      │
  │                                      │
  │─────── Unsubscribe ───────────────────→
  │                                      │
  │─────── Close connection ──────────────→
```

### Redis Pub/Sub

```
Rails Instance 1          Redis           Rails Instance 2
       │                    │                    │
       │──── Publish ───────→                    │
       │    (message)       │                    │
       │                    │──── Broadcast ─────→
       │                    │                    │
       │                    │                    │
       │←─── Subscribe ─────│                    │
       │    (to channel)    │                    │
```

## Security Architecture

### Authentication Flow
1. User submits username/password
2. Controller verifies via bcrypt
3. Session created with user_id
4. Session cookie stored (encrypted)
5. Subsequent requests authenticated via session

### WebSocket Security
1. Connection identified by session cookie
2. Only authenticated users can subscribe
3. User can only send to subscribed channels
4. Message content sanitized before broadcast

### CSRF Protection
- Rails automatic CSRF tokens
- Form submissions include authenticity_token
- WebSocket exempt (session-based auth)

## Deployment Architecture

### Development
```
localhost:3000 (Rails)
localhost:6379 (Redis)
PostgreSQL Docker container
```

### Production (Heroku)
```
Web Dyno (Rails app)
    ↓
Heroku Redis (Action Cable)
    ↓
Heroku Postgres (Database)
```

## Technology Stack

### Backend
- Ruby 3.2+
- Rails 7.2
- Action Cable (WebSocket)
- bcrypt (password hashing)
- Redis (pub/sub for Action Cable)

### Frontend
- Hotwire Turbo (Ajax navigation)
- Stimulus.js (lightweight JS framework)
- Tailwind CSS (utility-first CSS)
- Native browser WebSocket API

### Database
- PostgreSQL (primary datastore)
- Redis (WebSocket pub/sub)

### Infrastructure
- Docker (local PostgreSQL)
- Heroku or Render (deployment)
- CloudFlare (optional CDN)

## Scalability Considerations

### Current Capacity (MVP)
- 100+ concurrent users per room
- 1000+ total active connections
- Vertical scaling (larger dyno/server)

### Future Scaling (Post-MVP)
- Horizontal scaling with load balancer
- Separate Action Cable servers
- Redis cluster for pub/sub
- Database read replicas
- CDN for static assets

## Performance Optimizations

### Database
- Indexes on foreign keys (user_id, room_id)
- Pagination for message history
- Eager loading to prevent N+1 queries

### Caching
- Redis for Action Cable adapter
- HTTP caching headers
- Fragment caching for room list

### WebSocket
- Efficient message serialization (JSON)
- Debounce typing indicators (future)
- Batch presence updates

## Monitoring & Debugging

### Key Metrics
- WebSocket connection count
- Message delivery latency
- Database query time
- Redis memory usage
- Error rate

### Tools
- Rails logs (development)
- Heroku logs (production)
- Redis CLI for debugging
- Browser DevTools (Network, Console)

## Error Handling

### WebSocket Disconnection
- Automatic reconnection attempt
- Show connection status in UI
- Queue messages during disconnection
- Resend on reconnection

### Database Errors
- Retry transient failures
- Log errors for debugging
- Show user-friendly error messages
- Graceful degradation
