# Data Model

## Entity Relationship Diagram

```
┌─────────────────────┐
│       users         │
├─────────────────────┤
│ id (PK)             │
│ username (unique)   │
│ password_digest     │
│ created_at          │
│ updated_at          │
└─────────────────────┘
           │
           │ 1:N (has_many :messages)
           │
           ↓
┌─────────────────────┐         ┌─────────────────────┐
│      messages       │ N:1     │        rooms        │
├─────────────────────┤─────────├─────────────────────┤
│ id (PK)             │         │ id (PK)             │
│ content             │         │ name (unique)       │
│ user_id (FK)        │         │ description         │
│ room_id (FK)        │←────────│ created_at          │
│ created_at          │   1:N   │ updated_at          │
│ updated_at          │         └─────────────────────┘
└─────────────────────┘
```

## Table Definitions

### users
Stores user authentication information.

| Column           | Type         | Constraints              | Description                    |
|------------------|--------------|--------------------------|--------------------------------|
| id               | bigint       | PRIMARY KEY              | Auto-incrementing ID           |
| username         | string       | NOT NULL, UNIQUE         | Unique username for login      |
| password_digest  | string       | NOT NULL                 | Bcrypt hashed password         |
| created_at       | datetime     | NOT NULL                 | Account creation timestamp     |
| updated_at       | datetime     | NOT NULL                 | Last update timestamp          |

**Indexes:**
- Primary key on `id`
- Unique index on `username`

**Validations:**
- username: presence, uniqueness, length (3-20 characters)
- password: minimum 6 characters (on virtual attribute)

---

### rooms
Stores chat room information.

| Column       | Type         | Constraints              | Description                    |
|--------------|--------------|--------------------------|--------------------------------|
| id           | bigint       | PRIMARY KEY              | Auto-incrementing ID           |
| name         | string       | NOT NULL, UNIQUE         | Room display name              |
| description  | text         |                          | Optional room description      |
| created_at   | datetime     | NOT NULL                 | Room creation timestamp        |
| updated_at   | datetime     | NOT NULL                 | Last update timestamp          |

**Indexes:**
- Primary key on `id`
- Unique index on `name`

**Validations:**
- name: presence, uniqueness, length (1-50 characters)
- description: length (maximum 200 characters)

---

### messages
Stores chat messages sent in rooms.

| Column       | Type         | Constraints              | Description                    |
|--------------|--------------|--------------------------|--------------------------------|
| id           | bigint       | PRIMARY KEY              | Auto-incrementing ID           |
| content      | text         | NOT NULL                 | Message content                |
| user_id      | bigint       | NOT NULL, FOREIGN KEY    | References users.id            |
| room_id      | bigint       | NOT NULL, FOREIGN KEY    | References rooms.id            |
| created_at   | datetime     | NOT NULL                 | Message sent timestamp         |
| updated_at   | datetime     | NOT NULL                 | Last update timestamp          |

**Indexes:**
- Primary key on `id`
- Foreign key index on `user_id`
- Foreign key index on `room_id`
- Composite index on `(room_id, created_at)` for efficient history queries

**Validations:**
- content: presence, length (1-1000 characters)
- user_id: presence
- room_id: presence

**Associations:**
- belongs_to :user
- belongs_to :room

---

## Model Associations

### User Model
```ruby
class User < ApplicationRecord
  has_secure_password
  
  has_many :messages, dependent: :destroy
  
  validates :username, presence: true, 
                       uniqueness: true, 
                       length: { in: 3..20 }
  validates :password, length: { minimum: 6 }, allow_nil: true
end
```

### Room Model
```ruby
class Room < ApplicationRecord
  has_many :messages, dependent: :destroy
  
  validates :name, presence: true, 
                   uniqueness: true, 
                   length: { in: 1..50 }
  validates :description, length: { maximum: 200 }
  
  # Returns the 100 most recent messages
  def recent_messages(limit = 100)
    messages.includes(:user).order(created_at: :desc).limit(limit).reverse
  end
end
```

### Message Model
```ruby
class Message < ApplicationRecord
  belongs_to :user
  belongs_to :room
  
  validates :content, presence: true, length: { in: 1..1000 }
  
  # Broadcast message to room subscribers
  after_create_commit do
    broadcast_append_to(
      "room_#{room_id}",
      target: "messages",
      partial: "messages/message",
      locals: { message: self }
    )
  end
end
```

## Database Migrations

### Create Users Table
```ruby
class CreateUsers < ActiveRecord::Migration[7.2]
  def change
    create_table :users do |t|
      t.string :username, null: false
      t.string :password_digest, null: false

      t.timestamps
    end
    
    add_index :users, :username, unique: true
  end
end
```

### Create Rooms Table
```ruby
class CreateRooms < ActiveRecord::Migration[7.2]
  def change
    create_table :rooms do |t|
      t.string :name, null: false
      t.text :description

      t.timestamps
    end
    
    add_index :rooms, :name, unique: true
  end
end
```

### Create Messages Table
```ruby
class CreateMessages < ActiveRecord::Migration[7.2]
  def change
    create_table :messages do |t|
      t.text :content, null: false
      t.references :user, null: false, foreign_key: true
      t.references :room, null: false, foreign_key: true

      t.timestamps
    end
    
    add_index :messages, [:room_id, :created_at]
  end
end
```

## Sample Data (Seeds)

```ruby
# Create sample users
alice = User.create!(
  username: "alice",
  password: "password123"
)

bob = User.create!(
  username: "bob",
  password: "password123"
)

# Create sample rooms
general = Room.create!(
  name: "General",
  description: "General discussion room"
)

random = Room.create!(
  name: "Random",
  description: "Random topics and off-topic chat"
)

rails = Room.create!(
  name: "Rails Help",
  description: "Get help with Ruby on Rails development"
)

# Create sample messages
Message.create!(
  user: alice,
  room: general,
  content: "Welcome to the General chat room!"
)

Message.create!(
  user: bob,
  room: general,
  content: "Thanks! Happy to be here."
)
```

## Query Patterns

### Common Queries

**Get all rooms with message count:**
```ruby
Room.left_joins(:messages)
    .group(:id)
    .select('rooms.*, COUNT(messages.id) as messages_count')
    .order(created_at: :desc)
```

**Get recent messages for a room:**
```ruby
room.messages
    .includes(:user)
    .order(created_at: :desc)
    .limit(100)
    .reverse
```

**Get user's message count:**
```ruby
user.messages.count
```

**Find room by name:**
```ruby
Room.find_by(name: "General")
```

## Performance Considerations

### Indexes Strategy
- **Primary keys**: Automatic B-tree index for fast lookups
- **Foreign keys**: Indexed for efficient joins
- **Unique constraints**: Username and room name for data integrity
- **Composite index**: (room_id, created_at) for paginated message queries

### N+1 Query Prevention
```ruby
# ❌ Bad - N+1 queries
messages.each { |m| puts m.user.username }

# ✅ Good - Eager loading
messages.includes(:user).each { |m| puts m.user.username }
```

### Pagination
```ruby
# Use kaminari or pagy gem
messages.order(created_at: :desc).page(params[:page]).per(50)
```

## Data Integrity

### Foreign Key Constraints
- `messages.user_id` → `users.id` (CASCADE on delete)
- `messages.room_id` → `rooms.id` (CASCADE on delete)

### Dependent Destroy
- When a user is deleted, all their messages are deleted
- When a room is deleted, all its messages are deleted

### Validations
- Username uniqueness at database and application level
- Room name uniqueness at database and application level
- Message content presence
- Foreign key presence

## Future Enhancements

### Additional Tables (Post-MVP)

**room_memberships** (for private rooms)
```ruby
create_table :room_memberships do |t|
  t.references :user, null: false, foreign_key: true
  t.references :room, null: false, foreign_key: true
  t.string :role, default: "member" # member, moderator, admin
  t.timestamps
end
```

**reactions** (for message reactions)
```ruby
create_table :reactions do |t|
  t.references :user, null: false, foreign_key: true
  t.references :message, null: false, foreign_key: true
  t.string :emoji, null: false
  t.timestamps
end
```

**attachments** (for file uploads)
```ruby
create_table :active_storage_attachments do |t|
  # Rails Active Storage tables
end
```
