# Agent Usage Guide

## Overview
This guide explains how AI agents (like Claude, GPT-4, Gemini) should approach implementing this Rails chat application blueprint.

---

## Before Starting

### 1. Read Documents in Order
Read the blueprint documentation in this sequence:
1. **README.md** - Overview and quick start
2. **01_requirements.md** - What needs to be built
3. **02_architecture.md** - System design and components
4. **03_data_model.md** - Database schema
5. **04_user_flows.md** - User interactions
6. **This file (05_agent_usage.md)** - Implementation guidance
7. **06_tasks.md** - Step-by-step tasks
8. **99_agent_notes.md** - Tips and gotchas

### 2. Confirm Environment
Verify the following are available:
- Ruby 3.2+ installed
- Rails 7.2+ gem installed
- PostgreSQL running (Docker container)
- Redis running (for Action Cable)
- Node.js 18+ (for JavaScript)
- VS Code editor

### 3. Understand the Context
- This is a **real-time chat application**
- Uses **Action Cable** for WebSocket communication
- Focus on **simplicity** over feature completeness
- Target: **MVP in 4-6 hours**

---

## Implementation Strategy

### Phase 1: Foundation (Task 1-2)
**Goal:** Get Rails app running with authentication

1. Create new Rails app with proper configuration
2. Set up PostgreSQL connection
3. Configure Redis for Action Cable
4. Add required gems
5. Implement user authentication (bcrypt)
6. Create sign up / sign in / sign out flows

**Checkpoint:** Users can register and log in

### Phase 2: Core Models (Task 3-5)
**Goal:** Database and models ready

1. Generate models (User, Room, Message)
2. Set up associations
3. Add validations
4. Create seed data
5. Test model relationships in console

**Checkpoint:** Can create rooms and messages in Rails console

### Phase 3: Room Management (Task 6-7)
**Goal:** Users can browse and join rooms

1. Create RoomsController
2. Build room list view
3. Build room show view (chat page)
4. Add room creation form
5. Style with Tailwind CSS

**Checkpoint:** Users can see and create rooms (no real-time yet)

### Phase 4: Real-time Chat (Task 8-9)
**Goal:** WebSocket messaging works

1. Generate RoomChannel
2. Configure Action Cable
3. Create Stimulus controller for chat
4. Implement message broadcasting
5. Test WebSocket connection

**Checkpoint:** Messages appear in real-time across browsers

### Phase 5: User Presence (Task 10)
**Goal:** Show who's online

1. Track user subscriptions
2. Broadcast join/leave events
3. Update UI with online users
4. Test presence detection

**Checkpoint:** User list updates in real-time

### Phase 6: Polish (Task 11-12)
**Goal:** Production-ready features

1. Auto-scroll to latest message
2. Empty states
3. Loading indicators
4. Error handling
5. Mobile responsiveness
6. Final testing

**Checkpoint:** App works smoothly on mobile and desktop

---

## Working with Action Cable

### Key Concepts

**1. Channels are like WebSocket endpoints**
```ruby
# Think of channels as controllers for WebSocket
class RoomChannel < ApplicationCable::Channel
  def subscribed
    # User connects to a room
    stream_from "room_#{params[:room_id]}"
  end
  
  def unsubscribed
    # User disconnects
  end
end
```

**2. Broadcasting sends data to subscribers**
```ruby
# In your controller or model
ActionCable.server.broadcast(
  "room_#{room_id}",
  message: "Hello!"
)
```

**3. Client receives via JavaScript**
```javascript
// Stimulus controller
consumer.subscriptions.create(
  { channel: "RoomChannel", room_id: roomId },
  {
    received(data) {
      // Handle incoming message
      this.appendMessage(data)
    }
  }
)
```

### Common Patterns

**Pattern 1: Message Broadcasting**
```ruby
# After creating a message
message.broadcast_append_to(
  "room_#{room_id}",
  target: "messages",
  partial: "messages/message"
)
```

**Pattern 2: Presence Tracking**
```ruby
# Track connected users in Redis
def subscribed
  current_user.appear_in_room(params[:room_id])
  stream_from "room_#{params[:room_id]}"
end

def unsubscribed
  current_user.disappear_from_room(params[:room_id])
end
```

**Pattern 3: Authorized Subscriptions**
```ruby
# Only logged-in users can subscribe
class ApplicationCable::Connection < ActionCable::Connection::Base
  identified_by :current_user
  
  def connect
    self.current_user = find_verified_user
  end
  
  private
  
  def find_verified_user
    if verified_user = User.find_by(id: cookies.encrypted[:user_id])
      verified_user
    else
      reject_unauthorized_connection
    end
  end
end
```

---

## Code Organization

### File Structure
```
app/
├── channels/
│   ├── application_cable/
│   │   ├── channel.rb
│   │   └── connection.rb
│   └── room_channel.rb
├── controllers/
│   ├── application_controller.rb
│   ├── rooms_controller.rb
│   ├── messages_controller.rb
│   ├── users_controller.rb
│   └── sessions_controller.rb
├── models/
│   ├── user.rb
│   ├── room.rb
│   └── message.rb
├── views/
│   ├── layouts/
│   │   └── application.html.erb
│   ├── rooms/
│   │   ├── index.html.erb
│   │   └── show.html.erb
│   ├── messages/
│   │   └── _message.html.erb
│   ├── users/
│   │   └── new.html.erb
│   └── sessions/
│       └── new.html.erb
└── javascript/
    ├── controllers/
    │   └── room_controller.js
    └── channels/
        ├── consumer.js
        └── room_channel.js
```

### Controller Responsibilities

**RoomsController**
- index: List all rooms
- show: Display chat room
- create: Create new room

**MessagesController**
- create: Post new message (broadcasts via cable)

**UsersController**
- new: Sign up form
- create: Register user

**SessionsController**
- new: Sign in form
- create: Log in user
- destroy: Log out user

---

## Testing Strategy

### Manual Testing Checklist

**Authentication:**
- [ ] Can sign up with valid credentials
- [ ] Cannot sign up with invalid data
- [ ] Can sign in with correct password
- [ ] Cannot sign in with wrong password
- [ ] Can sign out
- [ ] Protected pages redirect when not logged in

**Room Management:**
- [ ] Can view room list
- [ ] Can create new room
- [ ] Cannot create room with duplicate name
- [ ] Can navigate to room

**Real-time Messaging:**
- [ ] Open room in 2+ browsers
- [ ] Send message in one browser
- [ ] Message appears in all browsers instantly
- [ ] Username and timestamp displayed
- [ ] Message persists after page refresh

**User Presence:**
- [ ] User appears in online list when joining
- [ ] User disappears when leaving
- [ ] Join notification visible to others
- [ ] Leave notification visible to others

**UI/UX:**
- [ ] Works on mobile screen sizes
- [ ] Auto-scrolls to latest message
- [ ] Input clears after sending
- [ ] Loading states visible
- [ ] Error messages displayed

### Multi-window Testing
```bash
# Terminal 1: Start Rails
rails server

# Terminal 2: Start Redis
redis-server

# Browser 1: Sign in as Alice
# Browser 2: Sign in as Bob (incognito mode)
# Both join the same room and chat
```

---

## Common Pitfalls

### 1. Redis Not Running
**Problem:** WebSocket connects but messages don't broadcast
**Solution:** Ensure Redis is running (`redis-server`)

### 2. Action Cable Not Configured
**Problem:** Cable connection fails
**Solution:** Check `config/cable.yml` has correct Redis URL

### 3. CSRF Token Issues
**Problem:** Form submissions fail
**Solution:** Ensure `<%= csrf_meta_tags %>` in layout

### 4. N+1 Queries
**Problem:** Slow message loading
**Solution:** Use `.includes(:user)` when loading messages

### 5. WebSocket Origin Errors
**Problem:** Cable rejected in production
**Solution:** Configure `config.action_cable.allowed_request_origins`

### 6. Memory Leaks
**Problem:** Multiple subscriptions created
**Solution:** Unsubscribe when leaving room

---

## Debugging Tips

### Check WebSocket Connection
```javascript
// In browser console
App.cable.connection.webSocket.readyState
// 1 = OPEN, 0 = CONNECTING, 2 = CLOSING, 3 = CLOSED
```

### Monitor Action Cable
```bash
# Rails console
ActionCable.server.connections.size
# Shows number of active connections
```

### Redis Debugging
```bash
# Redis CLI
redis-cli
> KEYS *
> GET action_cable:...
```

### Rails Console Testing
```ruby
# Test broadcasting manually
ActionCable.server.broadcast("room_1", { message: "test" })

# Check subscriptions
ActionCable.server.pubsub.send(:redis_connection).pubsub("channels", "*")
```

---

## Performance Optimization

### Database
- Add indexes on foreign keys
- Eager load associations (`.includes(:user)`)
- Limit message history (last 100 messages)

### Action Cable
- Use Redis adapter (not async in production)
- Set proper connection pool size
- Configure timeout settings

### Frontend
- Debounce scroll events
- Lazy load message history
- Use Turbo for fast navigation

---

## Deployment Considerations

### Heroku
```bash
# Add Redis addon
heroku addons:create heroku-redis:mini

# Set Action Cable URL
heroku config:set ACTION_CABLE_URL=wss://yourapp.herokuapp.com/cable
```

### Environment Variables
```bash
# .env file (use dotenv gem)
DATABASE_URL=postgresql://...
REDIS_URL=redis://localhost:6379/0
ACTION_CABLE_ALLOWED_ORIGINS=http://localhost:3000
```

### Production Checklist
- [ ] Redis addon provisioned
- [ ] Action Cable URL configured
- [ ] Allowed origins set
- [ ] Database migrations run
- [ ] Seed data loaded
- [ ] WebSocket working across browsers

---

## Agent Workflow

### Step-by-Step Process

1. **Read all documentation** (15 min)
   - Understand requirements and architecture
   - Note key features and constraints

2. **Set up Rails project** (30 min)
   - Create app with proper config
   - Add gems
   - Configure database and Redis

3. **Implement authentication** (45 min)
   - User model with bcrypt
   - Sign up / sign in / sign out

4. **Build core models** (30 min)
   - Room and Message models
   - Associations and validations
   - Seed data

5. **Create room views** (45 min)
   - Room list page
   - Room show page (chat)
   - Forms and navigation

6. **Add Action Cable** (60 min)
   - Generate channel
   - Configure connection
   - Implement broadcasting
   - JavaScript subscription

7. **Implement presence** (45 min)
   - Track join/leave
   - Update user list
   - Notifications

8. **Polish and test** (45 min)
   - Auto-scroll
   - Mobile responsive
   - Error handling
   - Multi-browser testing

**Total: 4-6 hours**

---

## Questions to Ask User

### Before Starting
- Do you have PostgreSQL running?
- Do you have Redis installed?
- Should we create a new Rails app or add to existing?
- Any specific room features needed?

### During Implementation
- Does the authentication flow work for you?
- Can you test in multiple browsers?
- Is the UI responsive enough on mobile?

### After Completion
- Should we add direct messages?
- Want file upload support?
- Need typing indicators?

---

## Success Criteria

### Definition of Done
A task is complete when:
1. ✅ Code written and working
2. ✅ No console errors
3. ✅ Tested manually
4. ✅ UI looks good
5. ✅ Real-time features work across browsers

### MVP Complete When
- [ ] Users can sign up and log in
- [ ] Users can create and join rooms
- [ ] Messages broadcast in real-time
- [ ] User presence visible
- [ ] No major bugs
- [ ] Mobile responsive
- [ ] WebSocket reconnects gracefully

---

## Next Steps After MVP

### Phase 2 Features
- Direct messaging (DM)
- File uploads
- Message editing/deletion
- Typing indicators
- Read receipts

### Phase 3 Features
- User profiles with avatars
- Private rooms
- Room moderation
- Search messages
- Email notifications

### Scaling
- Separate Action Cable servers
- Redis cluster
- Database read replicas
- CDN for assets
