# Implementation Tasks

Follow these tasks sequentially to build the Rails chat application with Action Cable.

---

## Task 1: Project Setup

### Create Rails Application
```bash
rails new chat-app \
  --database=postgresql \
  --css=tailwind \
  --javascript=importmap
```

### Configure Database
Edit `config/database.yml`:
```yaml
development:
  adapter: postgresql
  encoding: unicode
  database: chat_app_development
  pool: <%= ENV.fetch("RAILS_MAX_THREADS") { 5 } %>
  username: postgres
  password: postgres
  host: database-server-dev1
```

### Add Required Gems
Edit `Gemfile`:
```ruby
gem 'bcrypt', '~> 3.1.7'  # Uncomment if commented
gem 'redis', '~> 5.0'

group :development, :test do
  gem 'debug', platforms: %i[ mri windows ]
end
```

Run:
```bash
bundle install
```

### Configure Redis for Action Cable
Edit `config/cable.yml`:
```yaml
development:
  adapter: redis
  url: <%= ENV.fetch("REDIS_URL") { "redis://localhost:6379/1" } %>

test:
  adapter: test

production:
  adapter: redis
  url: <%= ENV.fetch("REDIS_URL") { "redis://localhost:6379/1" } %>
  channel_prefix: chat_app_production
```

### Setup Database
```bash
rails db:create
rails db:migrate
```

### Test Server
```bash
./bin/dev  # Or rails server
```

Visit http://localhost:3000 - you should see Rails welcome page.

**✅ Checkpoint:** Rails app running with PostgreSQL and Redis configured

---

## Task 2: User Authentication

### Generate User Model
```bash
rails generate model User username:string password_digest:string
rails db:migrate
```

### Configure User Model
Edit `app/models/user.rb`:
```ruby
class User < ApplicationRecord
  has_secure_password
  
  validates :username, presence: true, 
                       uniqueness: true, 
                       length: { in: 3..20 }
  validates :password, length: { minimum: 6 }, allow_nil: true
end
```

### Create Sessions Controller
```bash
rails generate controller Sessions new create destroy
```

Edit `app/controllers/sessions_controller.rb`:
```ruby
class SessionsController < ApplicationController
  def new
    # Sign in form
  end
  
  def create
    user = User.find_by(username: params[:username])
    if user&.authenticate(params[:password])
      session[:user_id] = user.id
      redirect_to rooms_path, notice: "Signed in successfully!"
    else
      flash.now[:alert] = "Invalid username or password"
      render :new, status: :unprocessable_entity
    end
  end
  
  def destroy
    session[:user_id] = nil
    redirect_to sign_in_path, notice: "Signed out successfully!"
  end
end
```

### Create Users Controller
```bash
rails generate controller Users new create
```

Edit `app/controllers/users_controller.rb`:
```ruby
class UsersController < ApplicationController
  def new
    @user = User.new
  end
  
  def create
    @user = User.new(user_params)
    if @user.save
      session[:user_id] = @user.id
      redirect_to rooms_path, notice: "Account created successfully!"
    else
      render :new, status: :unprocessable_entity
    end
  end
  
  private
  
  def user_params
    params.require(:user).permit(:username, :password, :password_confirmation)
  end
end
```

### Add Authentication Helpers
Edit `app/controllers/application_controller.rb`:
```ruby
class ApplicationController < ActionController::Base
  helper_method :current_user, :logged_in?
  
  def current_user
    @current_user ||= User.find_by(id: session[:user_id]) if session[:user_id]
  end
  
  def logged_in?
    current_user.present?
  end
  
  def require_login
    unless logged_in?
      flash[:alert] = "You must be logged in to access this page"
      redirect_to sign_in_path
    end
  end
end
```

### Create Routes
Edit `config/routes.rb`:
```ruby
Rails.application.routes.draw do
  root "sessions#new"
  
  get "sign_in", to: "sessions#new"
  post "sign_in", to: "sessions#create"
  delete "sign_out", to: "sessions#destroy"
  
  get "sign_up", to: "users#new"
  post "sign_up", to: "users#create"
  
  resources :rooms, only: [:index, :show, :create] do
    resources :messages, only: [:create]
  end
end
```

### Create Sign In View
Create `app/views/sessions/new.html.erb`:
```erb
<div class="max-w-md mx-auto mt-8">
  <h1 class="text-3xl font-bold mb-6">Sign In</h1>
  
  <%= form_with url: sign_in_path, method: :post, local: true do |f| %>
    <div class="mb-4">
      <%= f.label :username, class: "block mb-2" %>
      <%= f.text_field :username, required: true, 
          class: "w-full px-4 py-2 border rounded" %>
    </div>
    
    <div class="mb-4">
      <%= f.label :password, class: "block mb-2" %>
      <%= f.password_field :password, required: true, 
          class: "w-full px-4 py-2 border rounded" %>
    </div>
    
    <%= f.submit "Sign In", class: "w-full bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600" %>
  <% end %>
  
  <p class="mt-4 text-center">
    Don't have an account? <%= link_to "Sign Up", sign_up_path, class: "text-blue-500" %>
  </p>
</div>
```

### Create Sign Up View
Create `app/views/users/new.html.erb`:
```erb
<div class="max-w-md mx-auto mt-8">
  <h1 class="text-3xl font-bold mb-6">Sign Up</h1>
  
  <%= form_with model: @user, url: sign_up_path, local: true do |f| %>
    <% if @user.errors.any? %>
      <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
        <ul>
          <% @user.errors.full_messages.each do |message| %>
            <li><%= message %></li>
          <% end %>
        </ul>
      </div>
    <% end %>
    
    <div class="mb-4">
      <%= f.label :username, class: "block mb-2" %>
      <%= f.text_field :username, required: true, 
          class: "w-full px-4 py-2 border rounded" %>
    </div>
    
    <div class="mb-4">
      <%= f.label :password, class: "block mb-2" %>
      <%= f.password_field :password, required: true, 
          class: "w-full px-4 py-2 border rounded" %>
    </div>
    
    <div class="mb-4">
      <%= f.label :password_confirmation, class: "block mb-2" %>
      <%= f.password_field :password_confirmation, required: true, 
          class: "w-full px-4 py-2 border rounded" %>
    </div>
    
    <%= f.submit "Sign Up", class: "w-full bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600" %>
  <% end %>
  
  <p class="mt-4 text-center">
    Already have an account? <%= link_to "Sign In", sign_in_path, class: "text-blue-500" %>
  </p>
</div>
```

### Update Application Layout
Edit `app/views/layouts/application.html.erb`:
```erb
<!DOCTYPE html>
<html>
  <head>
    <title>ChatApp</title>
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <%= csrf_meta_tags %>
    <%= csp_meta_tag %>
    <%= stylesheet_link_tag "tailwind", "inter-font", "data-turbo-track": "reload" %>
    <%= stylesheet_link_tag "application", "data-turbo-track": "reload" %>
    <%= javascript_importmap_tags %>
  </head>

  <body class="bg-gray-50">
    <% if logged_in? %>
      <nav class="bg-white shadow-sm mb-6">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
          <h1 class="text-xl font-bold">💬 Chat App</h1>
          <div class="flex items-center space-x-4">
            <span>👤 <%= current_user.username %></span>
            <%= button_to "Sign Out", sign_out_path, method: :delete, 
                class: "bg-red-500 text-white px-4 py-2 rounded hover:bg-red-600" %>
          </div>
        </div>
      </nav>
    <% end %>
    
    <% if flash[:notice] %>
      <div class="container mx-auto px-4 mb-4">
        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded">
          <%= flash[:notice] %>
        </div>
      </div>
    <% end %>
    
    <% if flash[:alert] %>
      <div class="container mx-auto px-4 mb-4">
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
          <%= flash[:alert] %>
        </div>
      </div>
    <% end %>
    
    <%= yield %>
  </body>
</html>
```

**✅ Checkpoint:** Users can sign up and sign in

---

## Task 3: Create Room and Message Models

### Generate Room Model
```bash
rails generate model Room name:string description:text
```

### Generate Message Model
```bash
rails generate model Message content:text user:references room:references
```

### Add Index to Messages
Edit the migration file for messages and add:
```ruby
add_index :messages, [:room_id, :created_at]
```

### Run Migrations
```bash
rails db:migrate
```

### Configure Room Model
Edit `app/models/room.rb`:
```ruby
class Room < ApplicationRecord
  has_many :messages, dependent: :destroy
  
  validates :name, presence: true, 
                   uniqueness: true, 
                   length: { in: 1..50 }
  validates :description, length: { maximum: 200 }
  
  def recent_messages(limit = 100)
    messages.includes(:user).order(created_at: :desc).limit(limit).reverse
  end
end
```

### Configure Message Model
Edit `app/models/message.rb`:
```ruby
class Message < ApplicationRecord
  belongs_to :user
  belongs_to :room
  
  validates :content, presence: true, length: { in: 1..1000 }
  
  after_create_commit do
    broadcast_append_to(
      "room_#{room_id}",
      target: "messages",
      partial: "messages/message",
      locals: { message: self }
    )
  end
end
```

### Update User Model
Edit `app/models/user.rb` and add:
```ruby
has_many :messages, dependent: :destroy
```

**✅ Checkpoint:** Models created with associations

---

## Task 4: Create Seed Data

Edit `db/seeds.rb`:
```ruby
# Clear existing data
Message.destroy_all
Room.destroy_all
User.destroy_all

# Create users
alice = User.create!(username: "alice", password: "password123", password_confirmation: "password123")
bob = User.create!(username: "bob", password: "password123", password_confirmation: "password123")

puts "Created users: alice, bob (password: password123)"

# Create rooms
general = Room.create!(name: "General", description: "General discussion room")
random = Room.create!(name: "Random", description: "Random topics and off-topic chat")
rails_help = Room.create!(name: "Rails Help", description: "Get help with Ruby on Rails")

puts "Created rooms: General, Random, Rails Help"

# Create sample messages
Message.create!(user: alice, room: general, content: "Welcome to the General chat room!")
Message.create!(user: bob, room: general, content: "Thanks! Happy to be here.")
Message.create!(user: alice, room: rails_help, content: "Anyone have experience with Action Cable?")

puts "Created sample messages"
puts "✅ Seed data complete!"
```

Run seeds:
```bash
rails db:seed
```

**✅ Checkpoint:** Database has sample data

---

## Task 5: Create Rooms Controller and Views

### Generate Rooms Controller
```bash
rails generate controller Rooms index show create
```

### Configure Rooms Controller
Edit `app/controllers/rooms_controller.rb`:
```ruby
class RoomsController < ApplicationController
  before_action :require_login
  
  def index
    @rooms = Room.all.order(created_at: :desc)
    @room = Room.new
  end
  
  def show
    @room = Room.find(params[:id])
    @messages = @room.recent_messages
    @message = Message.new
  end
  
  def create
    @room = Room.new(room_params)
    if @room.save
      redirect_to @room, notice: "Room created successfully!"
    else
      @rooms = Room.all
      render :index, status: :unprocessable_entity
    end
  end
  
  private
  
  def room_params
    params.require(:room).permit(:name, :description)
  end
end
```

### Create Room Index View
Create `app/views/rooms/index.html.erb`:
```erb
<div class="container mx-auto px-4">
  <div class="grid md:grid-cols-3 gap-6">
    <!-- Room List -->
    <div class="md:col-span-2">
      <h2 class="text-2xl font-bold mb-4">Chat Rooms</h2>
      
      <div class="grid gap-4">
        <% @rooms.each do |room| %>
          <%= link_to room, class: "block bg-white p-6 rounded-lg shadow hover:shadow-lg transition" do %>
            <h3 class="text-xl font-bold mb-2"><%= room.name %></h3>
            <p class="text-gray-600"><%= room.description %></p>
            <p class="text-sm text-gray-500 mt-2">
              <%= room.messages.count %> messages
            </p>
          <% end %>
        <% end %>
      </div>
    </div>
    
    <!-- Create Room Form -->
    <div>
      <h2 class="text-2xl font-bold mb-4">Create Room</h2>
      
      <%= form_with model: @room, url: rooms_path, local: true, class: "bg-white p-6 rounded-lg shadow" do |f| %>
        <% if @room.errors.any? %>
          <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
            <ul>
              <% @room.errors.full_messages.each do |message| %>
                <li><%= message %></li>
              <% end %>
            </ul>
          </div>
        <% end %>
        
        <div class="mb-4">
          <%= f.label :name, class: "block mb-2 font-bold" %>
          <%= f.text_field :name, required: true, 
              class: "w-full px-4 py-2 border rounded" %>
        </div>
        
        <div class="mb-4">
          <%= f.label :description, class: "block mb-2 font-bold" %>
          <%= f.text_area :description, rows: 3, 
              class: "w-full px-4 py-2 border rounded" %>
        </div>
        
        <%= f.submit "Create Room", class: "w-full bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600" %>
      <% end %>
    </div>
  </div>
</div>
```

### Create Room Show View
Create `app/views/rooms/show.html.erb`:
```erb
<div class="container mx-auto px-4 max-w-4xl">
  <div class="bg-white rounded-lg shadow-lg">
    <!-- Room Header -->
    <div class="border-b p-4 flex justify-between items-center">
      <div>
        <h2 class="text-2xl font-bold"><%= @room.name %></h2>
        <p class="text-gray-600"><%= @room.description %></p>
      </div>
      <%= link_to "← Back to Rooms", rooms_path, 
          class: "text-blue-500 hover:text-blue-700" %>
    </div>
    
    <!-- Messages Container -->
    <div id="messages" class="p-4 h-96 overflow-y-auto space-y-3">
      <%= render @messages %>
    </div>
    
    <!-- Message Form -->
    <div class="border-t p-4">
      <%= form_with model: [@room, @message], 
                    local: false,
                    data: { controller: "reset-form", action: "turbo:submit-end->reset-form#reset" } do |f| %>
        <div class="flex space-x-2">
          <%= f.text_area :content, 
              rows: 2,
              placeholder: "Type your message...",
              required: true,
              class: "flex-1 px-4 py-2 border rounded resize-none" %>
          <%= f.submit "Send", 
              class: "bg-blue-500 text-white px-6 py-2 rounded hover:bg-blue-600" %>
        </div>
      <% end %>
    </div>
  </div>
</div>
```

### Create Message Partial
Create `app/views/messages/_message.html.erb`:
```erb
<div id="<%= dom_id(message) %>" class="flex items-start space-x-3">
  <div class="flex-1 bg-gray-100 rounded-lg p-3">
    <div class="flex items-baseline space-x-2 mb-1">
      <span class="font-bold text-sm"><%= message.user.username %></span>
      <span class="text-xs text-gray-500">
        <%= message.created_at.strftime("%I:%M %p") %>
      </span>
    </div>
    <p class="text-gray-800"><%= message.content %></p>
  </div>
</div>
```

**✅ Checkpoint:** Can view rooms and see message history

---

## Task 6: Create Messages Controller

Create `app/controllers/messages_controller.rb`:
```ruby
class MessagesController < ApplicationController
  before_action :require_login
  
  def create
    @room = Room.find(params[:room_id])
    @message = @room.messages.build(message_params)
    @message.user = current_user
    
    if @message.save
      # Message will be broadcast automatically via after_create_commit
      head :ok
    else
      render json: { errors: @message.errors.full_messages }, 
             status: :unprocessable_entity
    end
  end
  
  private
  
  def message_params
    params.require(:message).permit(:content)
  end
end
```

**✅ Checkpoint:** Can post messages (not real-time yet)

---

## Task 7: Configure Action Cable

### Create Room Channel
```bash
rails generate channel Room
```

### Configure Room Channel
Edit `app/channels/room_channel.rb`:
```ruby
class RoomChannel < ApplicationCable::Channel
  def subscribed
    room = Room.find(params[:room_id])
    stream_for room
  end

  def unsubscribed
    # Cleanup when channel is unsubscribed
  end
end
```

### Configure Cable Connection
Edit `app/channels/application_cable/connection.rb`:
```ruby
module ApplicationCable
  class Connection < ActionCable::Connection::Base
    identified_by :current_user

    def connect
      self.current_user = find_verified_user
    end

    private

    def find_verified_user
      if verified_user = User.find_by(id: cookies.encrypted[:user_id])
        verified_user
      else
        reject_unauthorized_connection
      end
    end
  end
end
```

### Create Stimulus Controller
Create `app/javascript/controllers/room_controller.js`:
```javascript
import { Controller } from "@hotwired/stimulus"
import { createConsumer } from "@rails/actioncable"

export default class extends Controller {
  static values = { roomId: Number }
  
  connect() {
    this.channel = createConsumer().subscriptions.create(
      { channel: "RoomChannel", room_id: this.roomIdValue },
      {
        connected: this._connected.bind(this),
        disconnected: this._disconnected.bind(this),
        received: this._received.bind(this)
      }
    )
    
    this.scrollToBottom()
  }
  
  disconnect() {
    this.channel.unsubscribe()
  }
  
  _connected() {
    console.log("Connected to room channel")
  }
  
  _disconnected() {
    console.log("Disconnected from room channel")
  }
  
  _received(data) {
    // Turbo Stream will handle the append
    this.scrollToBottom()
  }
  
  scrollToBottom() {
    const messages = document.getElementById("messages")
    if (messages) {
      messages.scrollTop = messages.scrollHeight
    }
  }
}
```

### Create Reset Form Controller
Create `app/javascript/controllers/reset_form_controller.js`:
```javascript
import { Controller } from "@hotwired/stimulus"

export default class extends Controller {
  reset() {
    this.element.reset()
  }
}
```

### Update Room Show View
Edit `app/views/rooms/show.html.erb` and add `data-controller`:
```erb
<div class="container mx-auto px-4 max-w-4xl" 
     data-controller="room" 
     data-room-room-id-value="<%= @room.id %>">
  <!-- rest of the code -->
</div>
```

**✅ Checkpoint:** Messages broadcast in real-time via WebSocket

---

## Task 8: Test Real-time Features

### Start Services
```bash
# Terminal 1: Start Redis
redis-server

# Terminal 2: Start Rails
./bin/dev
```

### Test in Multiple Browsers
1. Open http://localhost:3000 in Chrome
2. Sign in as alice
3. Join "General" room
4. Open http://localhost:3000 in Firefox (or Incognito)
5. Sign in as bob
6. Join "General" room
7. Send messages from both browsers
8. Messages should appear instantly in both windows

**✅ Checkpoint:** Real-time messaging works across browsers

---

## Task 9: Add Auto-scroll

The auto-scroll is already implemented in the Stimulus controller from Task 7.

Test:
1. Send multiple messages to fill the chat window
2. New messages should automatically scroll to bottom
3. If manually scrolled up, new messages still append (no auto-scroll)

**✅ Checkpoint:** Auto-scroll working

---

## Task 10: Mobile Responsiveness

Tailwind CSS handles most responsiveness. Test on mobile:
1. Open in Chrome DevTools mobile view
2. Test room list
3. Test chat interface
4. Ensure input is accessible above keyboard

Optional improvements in `app/views/rooms/show.html.erb`:
```erb
<!-- Make chat area taller on mobile -->
<div id="messages" class="p-4 h-[60vh] md:h-96 overflow-y-auto space-y-3">
```

**✅ Checkpoint:** Works on mobile devices

---

## Task 11: Error Handling

### Add Connection Status Indicator
Update `app/javascript/controllers/room_controller.js`:
```javascript
_connected() {
  console.log("Connected to room channel")
  this.showStatus("Connected", "green")
}

_disconnected() {
  console.log("Disconnected from room channel")
  this.showStatus("Disconnected", "red")
}

showStatus(message, color) {
  // Optional: Add status indicator to UI
}
```

### Handle Failed Message Send
Already handled in `messages_controller.rb` with validation errors.

**✅ Checkpoint:** Basic error handling in place

---

## Task 12: Final Testing

### Testing Checklist
- [ ] Sign up with valid data
- [ ] Sign up with invalid data (see errors)
- [ ] Sign in with correct password
- [ ] Sign in with wrong password
- [ ] View room list
- [ ] Create new room
- [ ] Join existing room
- [ ] See message history
- [ ] Send message (appears for sender)
- [ ] Open second browser, send message (appears in first browser)
- [ ] Messages persist after refresh
- [ ] Sign out works
- [ ] Mobile responsive
- [ ] No console errors

### Performance Check
- [ ] Page loads in < 2 seconds
- [ ] Messages appear in < 100ms
- [ ] No N+1 queries (check logs)
- [ ] Redis connected

**✅ Checkpoint:** All features working, MVP complete!

---

## Task 13: Optional Enhancements

### User Presence
Track online users in each room:

1. Add Stimulus controller to track presence
2. Broadcast join/leave events
3. Display online user list
4. Show join/leave notifications

### Typing Indicators
Show "User is typing..." indicator:

1. Debounce keystrokes
2. Broadcast typing event
3. Display indicator in UI
4. Clear after 3 seconds

### Message Timestamps
Better formatting:

```ruby
# In message partial
<%= time_ago_in_words(message.created_at) %> ago
```

---

## Deployment

### Heroku Deployment
```bash
heroku create your-chat-app
heroku addons:create heroku-postgresql:mini
heroku addons:create heroku-redis:mini
git push heroku main
heroku run rails db:migrate
heroku run rails db:seed
heroku open
```

### Environment Variables
```bash
heroku config:set ACTION_CABLE_ALLOWED_ORIGINS=https://your-app.herokuapp.com
```

---

## Maintenance Tasks

### Database Cleanup
```ruby
# Remove old messages (optional cron job)
Message.where("created_at < ?", 30.days.ago).destroy_all
```

### Monitor Connections
```ruby
# Rails console
ActionCable.server.connections.count
```

---

## Success Metrics

After completing all tasks:
- ✅ Users can register and authenticate
- ✅ Users can create and join rooms
- ✅ Messages broadcast in real-time
- ✅ WebSocket connection stable
- ✅ UI responsive on mobile
- ✅ No major bugs or errors
- ✅ Code organized and readable
- ✅ Ready for deployment

**🎉 Congratulations! Your Rails chat app with Action Cable is complete!**
