# Agent Notes

## Implementation Tips and Gotchas

This document contains practical notes, tips, and common issues encountered during implementation.

---

## Action Cable Gotchas

### 1. Redis Must Be Running
**Problem:** WebSocket connects but messages don't broadcast.

**Symptom:**
```
Connection successful but no messages appearing
```

**Solution:**
```bash
# Make sure Redis is running
redis-server

# Or in Docker
docker run -d -p 6379:6379 redis:7-alpine
```

**Verify:**
```bash
redis-cli ping
# Should return: PONG
```

---

### 2. Cookie-based Authentication
**Problem:** WebSocket connection rejected.

**Symptom:**
```
An unauthorized connection attempt was rejected
```

**Solution:** Ensure session cookie is set after login:
```ruby
# In SessionsController#create
session[:user_id] = user.id
```

**Important:** Action Cable uses the same session cookie as Rails, so authentication "just works" if session is properly set.

---

### 3. Turbo Streams vs Manual Broadcasting
**Problem:** Confused about when to use which.

**Best Practice:**
```ruby
# ✅ Use Turbo Streams (automatic)
class Message < ApplicationRecord
  after_create_commit do
    broadcast_append_to(
      "room_#{room_id}",
      target: "messages",
      partial: "messages/message"
    )
  end
end

# ❌ Don't manually broadcast HTML
ActionCable.server.broadcast("room_1", { html: render_message })
```

**Why:** Turbo Streams handle DOM updates automatically.

---

### 4. N+1 Queries in Real-time
**Problem:** Slow message rendering.

**Bad:**
```ruby
def recent_messages
  messages.order(created_at: :desc).limit(100)
end
```

**Good:**
```ruby
def recent_messages
  messages.includes(:user).order(created_at: :desc).limit(100)
end
```

**Verify in logs:**
```
# Bad (N+1)
SELECT * FROM messages WHERE room_id = 1
SELECT * FROM users WHERE id = 1
SELECT * FROM users WHERE id = 2
...

# Good (2 queries)
SELECT * FROM messages WHERE room_id = 1
SELECT * FROM users WHERE id IN (1, 2, 3, ...)
```

---

## Stimulus Controller Tips

### 1. Auto-scroll on New Message
**Implementation:**
```javascript
_received(data) {
  // Turbo Stream handles DOM update
  // Just scroll after a short delay
  setTimeout(() => this.scrollToBottom(), 50)
}

scrollToBottom() {
  const container = document.getElementById("messages")
  container.scrollTop = container.scrollHeight
}
```

**Why the delay:** Turbo Stream needs time to append DOM before scrolling.

---

### 2. Clear Input After Send
**Problem:** Input box doesn't clear after sending message.

**Solution:** Use Stimulus controller:
```javascript
// reset_form_controller.js
import { Controller } from "@hotwired/stimulus"

export default class extends Controller {
  reset(event) {
    if (event.detail.success) {
      this.element.reset()
    }
  }
}
```

**In form:**
```erb
<%= form_with model: [@room, @message],
              data: { 
                controller: "reset-form",
                action: "turbo:submit-end->reset-form#reset"
              } do |f| %>
```

---

### 3. Unsubscribe on Disconnect
**Problem:** Multiple subscriptions created when navigating.

**Solution:** Always unsubscribe:
```javascript
disconnect() {
  if (this.channel) {
    this.channel.unsubscribe()
  }
}
```

---

## Database Tips

### 1. Composite Index for Messages
**Why:**
```sql
-- Common query pattern
SELECT * FROM messages 
WHERE room_id = 1 
ORDER BY created_at DESC 
LIMIT 100;
```

**Migration:**
```ruby
add_index :messages, [:room_id, :created_at]
```

**Result:** 10x faster queries on large datasets.

---

### 2. Dependent Destroy
**Important:** Set up cascading deletes:
```ruby
class Room < ApplicationRecord
  has_many :messages, dependent: :destroy
end

class User < ApplicationRecord
  has_many :messages, dependent: :destroy
end
```

**Why:** Prevents orphaned records.

---

## Tailwind CSS Tips

### 1. Scrollable Chat Container
```erb
<div class="h-96 overflow-y-auto">
  <!-- messages -->
</div>
```

**Mobile-friendly:**
```erb
<div class="h-[60vh] md:h-96 overflow-y-auto">
```

---

### 2. Message Bubble Layout
```erb
<div class="flex items-start space-x-3">
  <div class="flex-1 bg-gray-100 rounded-lg p-3">
    <!-- message content -->
  </div>
</div>
```

**Why `flex-1`:** Allows message to expand but not overflow.

---

## Testing Tips

### 1. Multi-browser Testing
```bash
# Regular browser
open http://localhost:3000

# Incognito/Private
# Chrome: Cmd+Shift+N (Mac) / Ctrl+Shift+N (Windows)
# Firefox: Cmd+Shift+P (Mac) / Ctrl+Shift+P (Windows)
```

**Pro tip:** Use different user agents to see different usernames.

---

### 2. WebSocket Debugging
**Browser Console:**
```javascript
// Check connection status
App.cable.connection.webSocket.readyState
// 0 = CONNECTING, 1 = OPEN, 2 = CLOSING, 3 = CLOSED

// See active subscriptions
App.cable.subscriptions.subscriptions
```

**Rails Console:**
```ruby
# Check active connections
ActionCable.server.connections.count

# Broadcast manually
ActionCable.server.broadcast("room_1", { test: "message" })
```

---

### 3. Redis Debugging
```bash
redis-cli

# List all keys
KEYS *

# Monitor in real-time
MONITOR

# Get value
GET action_cable:...
```

---

## Performance Optimization

### 1. Limit Message History
**Don't load all messages:**
```ruby
# Bad
@messages = @room.messages.includes(:user)

# Good
@messages = @room.recent_messages(100)
```

---

### 2. Redis Connection Pool
**For production:**
```yaml
# config/cable.yml
production:
  adapter: redis
  url: <%= ENV['REDIS_URL'] %>
  channel_prefix: chat_app_production
  pool_size: <%= ENV.fetch("RAILS_MAX_THREADS") { 5 } %>
```

---

### 3. Compress Large Payloads
**For images/files (future feature):**
```ruby
# Use Active Storage with CDN
# Don't broadcast binary data via Action Cable
```

---

## Security Notes

### 1. Sanitize User Input
**Already handled by Rails:**
```erb
<!-- Automatic HTML escaping -->
<p><%= message.content %></p>
```

**For rich text (future):**
```ruby
gem 'sanitize'

# Sanitize before save
before_save :sanitize_content

def sanitize_content
  self.content = Sanitize.fragment(content, Sanitize::Config::BASIC)
end
```

---

### 2. Rate Limiting (Optional)
**Prevent spam:**
```ruby
# Gemfile
gem 'rack-attack'

# config/initializers/rack_attack.rb
Rack::Attack.throttle("messages/ip", limit: 10, period: 60) do |req|
  req.ip if req.path.start_with?('/rooms') && req.post?
end
```

---

### 3. Channel Authorization
**Already implemented:**
```ruby
# app/channels/application_cable/connection.rb
def connect
  self.current_user = find_verified_user
end

def find_verified_user
  if verified_user = User.find_by(id: cookies.encrypted[:user_id])
    verified_user
  else
    reject_unauthorized_connection
  end
end
```

**Result:** Only authenticated users can connect.

---

## Deployment Notes

### Heroku-specific

**1. Redis Addon:**
```bash
heroku addons:create heroku-redis:mini
```

**2. Action Cable Configuration:**
```ruby
# config/environments/production.rb
config.action_cable.url = ENV.fetch("ACTION_CABLE_URL") { 
  "wss://#{ENV['HEROKU_APP_NAME']}.herokuapp.com/cable" 
}
config.action_cable.allowed_request_origins = [
  "https://#{ENV['HEROKU_APP_NAME']}.herokuapp.com"
]
```

**3. Environment Variables:**
```bash
heroku config:set ACTION_CABLE_URL=wss://your-app.herokuapp.com/cable
```

---

### Database Performance

**1. Connection Pool:**
```yaml
# config/database.yml
production:
  pool: <%= ENV.fetch("RAILS_MAX_THREADS") { 5 } %>
```

**2. Query Timeout:**
```ruby
# config/database.yml
production:
  variables:
    statement_timeout: 5000  # 5 seconds
```

---

## Common Error Messages

### "Redis connection refused"
**Cause:** Redis not running  
**Fix:**
```bash
redis-server
```

---

### "Couldn't find Room with 'id'="
**Cause:** Missing room parameter or invalid ID  
**Fix:** Check routes and form submission

---

### "An unauthorized connection attempt was rejected"
**Cause:** User not logged in  
**Fix:** Ensure session[:user_id] is set

---

### "ActionController::InvalidAuthenticityToken"
**Cause:** Missing CSRF token  
**Fix:** Ensure `<%= csrf_meta_tags %>` in layout

---

## Future Enhancements

### 1. Direct Messages
**Schema:**
```ruby
create_table :conversations do |t|
  t.references :sender, foreign_key: { to_table: :users }
  t.references :receiver, foreign_key: { to_table: :users }
  t.timestamps
end

create_table :direct_messages do |t|
  t.references :conversation
  t.references :user
  t.text :content
  t.timestamps
end
```

---

### 2. Typing Indicators
**Channel method:**
```ruby
def typing
  broadcast_to(
    room,
    type: "typing",
    user: current_user.username
  )
end
```

**Client debounce:**
```javascript
typing = debounce(() => {
  this.channel.perform("typing")
}, 300)
```

---

### 3. Read Receipts
**Schema:**
```ruby
add_column :messages, :read_by, :jsonb, default: []
```

**Track:**
```ruby
def mark_as_read(user_id)
  update(read_by: read_by + [user_id])
end
```

---

### 4. Message Editing
**Schema:**
```ruby
add_column :messages, :edited_at, :datetime
```

**Broadcast update:**
```ruby
after_update_commit do
  broadcast_replace_to(
    "room_#{room_id}",
    target: self,
    partial: "messages/message"
  )
end
```

---

## Best Practices Summary

### Do's ✅
- Use Turbo Streams for DOM updates
- Eager load associations (`.includes(:user)`)
- Limit message history (100 messages max)
- Unsubscribe when leaving channel
- Test in multiple browsers
- Use Redis in production
- Set up proper indexes

### Don'ts ❌
- Don't manually manipulate DOM in channel callbacks
- Don't forget to start Redis
- Don't broadcast large payloads
- Don't skip authentication checks
- Don't load all messages at once
- Don't forget CSRF tokens
- Don't ignore N+1 queries

---

## Debugging Workflow

1. **Check Rails logs:**
   ```bash
   tail -f log/development.log
   ```

2. **Check Redis:**
   ```bash
   redis-cli ping
   ```

3. **Check browser console:**
   - Look for WebSocket connection
   - Check for JavaScript errors

4. **Test in Rails console:**
   ```ruby
   # Test models
   room = Room.first
   room.messages.create!(user: User.first, content: "test")
   
   # Test broadcasting
   ActionCable.server.broadcast("room_1", { message: "test" })
   ```

5. **Monitor Action Cable:**
   ```bash
   # Logs show connections and subscriptions
   Started GET "/cable" for 127.0.0.1
   Successfully upgraded to WebSocket
   ```

---

## Resources

### Official Docs
- [Action Cable Overview](https://guides.rubyonrails.org/action_cable_overview.html)
- [Turbo Streams](https://turbo.hotwired.dev/handbook/streams)
- [Stimulus Handbook](https://stimulus.hotwired.dev/handbook/introduction)

### Useful Gems
- `redis` - Redis client
- `bcrypt` - Password hashing
- `tailwindcss-rails` - Styling
- `turbo-rails` - Hotwire Turbo
- `stimulus-rails` - Stimulus JS

---

## Final Checklist

Before considering the project complete:

- [ ] Redis running and connected
- [ ] Users can sign up/in/out
- [ ] Rooms can be created
- [ ] Messages send in real-time
- [ ] Messages persist in database
- [ ] UI works on mobile
- [ ] No N+1 queries
- [ ] No console errors
- [ ] WebSocket reconnects automatically
- [ ] Code is clean and organized
- [ ] Ready for deployment

---

## Support & Troubleshooting

If you encounter issues:

1. Check this document first
2. Review error messages carefully
3. Test each component individually
4. Use Rails console for debugging
5. Check Redis and PostgreSQL are running
6. Verify browser WebSocket support

**Remember:** Most Action Cable issues are caused by:
- Redis not running
- Missing authentication
- N+1 queries
- Subscription not unsubscribing

---

**Good luck with your implementation! 🚀**
